/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/* -----------------------------------------------------------------------
 * File: session.h
 * Author: Joerg Striegel 4.6.96
 * Purpose: This file contains the function prototypes for the session 
 * ----------------------------------------------------------------------- */

#ifndef SESSION_H_INCLUDED
#define SESSION_H_INCLUDED

#include <propdefs.h>


#ifdef __cplusplus
extern "C" {
#endif


/* *********************************************************************
   Connections and opening. This functions do not exist in firmware
   ********************************************************************* */


#ifndef BEST_FIRMWARE
  
#define CONNECT_EXCLUSIVE  2
#define CONNECT_NORMAL     1



/* control functions */

void	BestConnectTimeoutSet (b_handletype handle, b_int32 ms);
void	BestConnectTimeoutDefSet (b_handletype handle);


/* -----------------------------------------------------------------------
 * Session Functions
 * ----------------------------------------------------------------------- */

/* BestDevIdentifierGet moved to mailbox.h */

b_errtype EXPORT BestOpen(
			  b_handletype  *handle,
			  b_porttype    port,
			  b_int32       portnum
			  );

/* BestOpenAdv

   !! undocumented function
   
   The underlaying open function that actually performs the task. All open
   functions call BestOpenAdv with an according parameter set. This function
   is for internal use only

   arguments:
   handle	- reference to the returned handle, result of this function
   port		- port for connection
   portnum	- details on port
   license_code - returns license information to the caller if the
   		  B_OPEN_LICENSE task is specified. In this case provide
		  an array of at least 3 longs. The first long returns the
		  board license, the other fields are reserved. NULL is legal.
   task		- special tasks during open.

   return value:
   B_E_OK
   B_E_CAPI_VERSION_CHECK
   B_E_HANDLE_NOT_OPEN
   B_E_NO_HANDLE_LEFT
   B_E_WRONG_PORT
   B_E_RS232_OPEN
   B_E_PARALLEL_OPEN
   B_E_PCI_OPEN
   B_E_IO_OPEN
*/

b_errtype EXPORT BestOpenAdv( /* advanced open call for GUI and failsafe */
			  b_handletype  *handle, /* open. BestOpen calls this*/
			  b_porttype    port, /* function. */
			  b_int32       portnum,
			  b_int32 *license_code,
			  b_OpenType task
			  );

b_errtype EXPORT BestClose(
			   b_handletype handle
			   );


b_errtype EXPORT BestCloseAdv(
			   b_handletype handle,
			   b_closetype	task
			   );

b_errtype EXPORT BestReopen ( /* closes and opens a card that was open */
                              b_handletype handle, /* does not change! */
			      b_OpenType   task
			    );

b_errtype BestFirstFreeHandleGet(
                                 b_handletype * pHandle
                                 );

b_errtype EXPORT BestBasicIsBoardReset(
                    b_handletype handle, 
                    b_int32 *fBoardWasReset);


CLI b_errtype EXPORT  BestRS232BaudRateSet(                             /* @brset */
					   b_handletype handle,
					   b_int32 baudrate             /* @baud */
					   );


b_errtype EXPORT BestDebug(
			   b_handletype handle,
			   b_int32      debug
			   );



b_errtype EXPORT BestGetDebugOutFromHandle( b_handletype handle,
					    void     (**std_out)  (char *,...) );


b_errtype EXPORT BestGetPortFromHandle( b_handletype handle,
					b_porttype   *port );

b_errtype EXPORT BestGetPortnumberFromHandle( b_handletype handle,
					      b_int32      *portnumber );

b_errtype EXPORT BestGetHWFromHandle( b_handletype handle,
				      b_hwtype     *hw );

b_errtype EXPORT BestGetProductstringFromHandle( b_handletype handle,
						 b_charptrtype     * pChar );

b_errtype EXPORT BestGetPciDevIdFromHandle(b_handletype handle,
                                           b_int32 * dev_id);

b_bool	  EXPORT BestIsOpen (b_handletype handle);
b_bool    EXPORT BestIsKnown (b_handletype handle);
b_bool    EXPORT BestIs2925 (b_handletype handle);
b_bool    EXPORT BestIs2925Deep (b_handletype handle);
b_bool    EXPORT BestHasFastHost(b_handletype handle);
b_bool    EXPORT BestHasIchiban(b_handletype handle);
b_bool    EXPORT BestHasIchiban20(b_handletype handle);
b_bool    EXPORT BestHasPerformance(b_handletype handle);
b_bool    EXPORT BestIsDeep(b_handletype handle);
b_bool    EXPORT BestIsCompact(b_handletype handle);
b_bool	  EXPORT BestIsCore (b_handletype handle);
b_bool    EXPORT BestIsInBlockMode(b_handletype handle);

void    BestHwBlockModeBitSet(b_handletype handle, b_bool fSet);

void    BestHwBitMaskSet(b_handletype handle, b_int32 mask, b_bool fSet);
b_bool  Best16BitRegisterFile (b_handletype handle);


/* --------------------------------------------------------------------------
 * Access (limited) to card types
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestGetCardTypeCount(size_t * pCount);

b_errtype EXPORT BestGetCardTypeAtIndex(
  b_param_stringlisttype * pCardList, 
  size_t idx
  );


/* How to find out if a PCI device id belongs to a (any) Best Card 
 */
b_bool  EXPORT BestIsPciDeviceIdBest(b_int32 deviceid);

/* The correct way of finding out what series of board you're
 * connected to
 */
b_hwseriestype EXPORT BestHwSeriesGet(b_handletype handle);

/* -----------------------------------------------------------------------
 * Connection Functions
 * ----------------------------------------------------------------------- */

CLI  b_errtype EXPORT BestConnect(                     /* @con */
				  b_handletype handle
				  );


CLI  b_errtype EXPORT BestDisconnect(                     /* @discon */
				     b_handletype handle
				     );

CLI  b_errtype EXPORT BestConnectEx(                     /* @conex */
				    b_handletype handle
				    );

CLI  b_errtype EXPORT BestDummySet (			/* @dummyset */
  	   		b_handletype handle,
			b_int32 value			/* @val */
  				   );

CLI  b_errtype EXPORT BestDummyGet (			/* @dummyget */
  	   		b_handletype handle,
			b_int32 *value			/* #RETURN "value=%08x" */
  				   );

CLI  b_errtype EXPORT BestPing (			/* @ping */
  			b_handletype handle
  			       	);

CLI  b_errtype EXPORT BestClockSet (			/* @clkset */
  			b_handletype handle,
			b_int32 clkvalue		/* @val */
			);

CLI  b_errtype EXPORT BestClockGet (			/* @clkset */
  			b_handletype handle,
			b_int32 *clkvalue		/* #RETURN "clock=%ds */
			);

#endif /* not firmware */

/* -----------------------------------------------------------------------
 * resource locking
 *
 * The following functions lock or unlock a requested resource to the
 * calling port. Only this port then has access rights for changing
 * settings of that port or start any action. Readings are allowed for all
 * ports if they do not have side effects. Lock/Unlock pairs can be nested
 * but must be balanced. BestUnlockAll unlocks all resources by force.
 *
 * return values:
 * B_EFW_OK
 * B_CMDLOCKED	- the resource is already locked from another port
 * ----------------------------------------------------------------------- */

CLI  b_errtype EXPORT BestResourceLock (	      /* @reslock */
    			b_handletype handle,
			b_resourcetype resource	      /* @res */
			       );

CLI  b_errtype EXPORT BestResourceUnlock (	   /* @resunlock */
    			b_handletype handle,
			b_resourcetype resource	   /* @res */
			        );

CLI  b_errtype EXPORT BestAllResourceUnlock (	  /* @allresunlock */
    			b_handletype handle
			        );

CLI  b_errtype EXPORT BestResourceIsLocked (	  /* @resislocked */
    			b_handletype handle,
                        b_resourcetype resource,  /* @res */
                        b_int32 *lock_count,     /* #RETURN "lock count=%d " */
			b_porttype *lock_port	  /* #RETURN "lock port=%s" */
			        );

/* -----------------------------------------------------------------------
 *
 * ----------------------------------------------------------------------- */

/* BestAbstractProp...

   New 16-bit abstract property set. A makro around the BestBaseCommand
   function to take advantage of the regular structure of property accesses.
   There are three versions of the get/set pair to transfer the values of
   a property group as condensed as possible.

   arguments:
   handle	- identifies port and hardware
   cmdcode	- 16-bit code identifying the called function
   prpid	- id that identifies the property, pass the value of the
   		  enum type
   prpval	- in/out, value of the property
*/

b_errtype BestAbstractPropByteSet (b_handletype		handle,
				   b_int16		cmdcode,
				   b_int8		prpid,
				   b_int32	        prpval);

b_errtype BestAbstractPropShortSet (b_handletype	handle,
				    b_int16		cmdcode,
				    b_int8		prpid,
				    b_int32	        prpval);

b_errtype BestAbstractPropLongSet (b_handletype		handle,
				   b_int16		cmdcode,
				   b_int8		prpid,
				   b_int32	        prpval);

b_errtype BestAbstractPropByteGet (b_handletype		handle,
				   b_int16		cmdcode,
				   b_int8		prpid,
				   b_int32	       *prpval);

b_errtype BestAbstractPropShortGet (b_handletype	handle,
				    b_int16		cmdcode,
				    b_int8		prpid,
				    b_int32	       *prpval);

b_errtype BestAbstractPropLongGet (b_handletype		handle,
				   b_int16		cmdcode,
				   b_int8		prpid,
				   b_int32	       *prpval);

/* 
 * SCR; this series of helper functions simplify calls to BestBasicCommand
 * which send OR receive (NOT both) simple variables.
 * NOTE; the IN_/OUT_ macros are not used....the stream lengths are implicit.
 */

b_errtype BBCSendByte(b_handletype handle, b_int16 cmdcode, b_int8ptr  pByte);
b_errtype BBCSendWord(b_handletype handle, b_int16 cmdcode, b_int16 * pUWord);
b_errtype BBCSendLong(b_handletype handle, b_int16 cmdcode, b_int32 * pULong);

b_errtype BBCReceiveByte(b_handletype handle, b_int16 cmdcode, b_int8ptr  pByte);
b_errtype BBCReceiveWord(b_handletype handle, b_int16 cmdcode, b_int16 * pUWord);
b_errtype BBCReceiveLong(b_handletype handle, b_int16 cmdcode, b_int32 * pULong);


/* Best...Copy

   These functions copy the contents of src, count times, into the destination buffer. 
   The functions eliminate the endianness problem. 
   
   Use these functions to translate data to and from the card.

   arguments:
   dest		- destination address
   src		- source address
   count		- number of distinct data items to copy from the source address to
              dest (each is sizeof(*src)); if src points to 10 longs then count = 10

   return values:
   incremented pointer to destination for subsequent calls of the copy
   functions.
   i.e. (dest + (count * sizeof(*src)))
*/

#ifdef BEST_FIRMWARE

#include <bytestr.h>

#define BestByteCopy(dest, src, count)		\
	(b_int8ptr)ByteStreamPutByte (dest, src, count)

#define BestWord2Stream(dest, src, count)	\
	(b_int8ptr)ByteStreamPutWord (dest, src, count)

#define BestLong2Stream(dest, src, count)	\
	(b_int8ptr)ByteStreamPutLong (dest, src, count)

#define BestStream2Word(dest, src, count)	\
	(b_int8ptr)ByteStreamGetWord (src, dest, count)

#define BestStream2Long(dest, src, count)	\
	(b_int8ptr)ByteStreamGetLong (src, dest, count)

	  
#else
b_int8ptr EXPORT BestByteCopy (b_int8ptr dest, b_int8ptr src, b_int64 count);

b_int8ptr EXPORT BestWord2Stream (b_int8ptr dest, b_int16ptr src, b_int32 count); /* num_words */
b_int8ptr EXPORT BestLong2Stream (b_int8ptr dest, b_int32ptr src, b_int32 count); /* num_dwords */

b_int8ptr EXPORT BestStream2Word (b_int16ptr dest, b_int8ptr src, b_int32 count); /* num_words */
b_int8ptr EXPORT BestStream2Long (b_int32ptr dest, b_int8ptr src, b_int32 count); /* num_dwords */

/* --------------------------------------------------------------------------
 * Provide a list of ALL cards currently attached to any port.
 * STORAGE for the B_DEVICE_DATA array is provided by this function.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestPortScan(
                              int * pNumInstances,
                              B_DEVICE_DATA ** ppDevArr
                              );


/* --------------------------------------------------------------------------
 * Provide data about one card connected to one port.
 * STORAGE for the B_DEVICE_DATA struct is provided by the CALLING function.
 * NOTE; This function does NOT flash the LED's on the card !!!
 *       This is not a full ping.
 * If there is no card connected to the port (i.e. the BestOpen() fails for
 * ANY reason) then we return B_E_CANNOT_CONNECT
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestPortPingQuick(
                              b_porttype port,           /* B_PORT_RS232 etc. */
                              unsigned long portnum,
                              B_DEVICE_DATA * pDevStruct
                              );

/* --------------------------------------------------------------------------
 * This function answers the question 
 * "Is the card that's connected to port/portnum also on the pci bus ?"
 * "Is the card with pSerialNum also on the pci bus ?"
 * NOTE; to save time you should be confident that there really is a card
 *       on the external port before calling this function.
 *       The caller must provide pSerialNum if a card is currently connected
 *       or port/portnum and pSerialNum=NULL if not.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestIsDutHost(
                          b_porttype port,          /* B_PORT_RS232 etc. */
                          unsigned long portnum,    /* 0,1 etc */
                          b_charptrtype pSerialNum, /* NULL => use port/portnum */
                          b_bool * fDutIsInHost     /* returns 0 or 1 */
                          );


#endif


#ifdef __cplusplus
}
#endif

#endif
